/*
 * RC5 remote control decoder library with LIRC support.
 *
 * Oct 2000        Clearwater Software
 * Guy Carpenter   guy@clearwater.com.au
 *
 * This code is released as-is and without warranty
 * for any use, commercial or non-commercial.
 *
 * For more information including an illustration of the
 * state machine see:
 *        http://www.clearwater.com.au/pcm-9574/rc5/rc5.html
 *
 * With LIRC, use as follows:
 *
 * Declare an RC5 context and initialize it:
 *     RC5_CONTEXT rc5;
 *     Rc5_Init(&rc5);
 * 
 * Repeatedly call Rc5_Notify with lirc data:
 *       if (Rc5_Notify(&rc5, data)) {
 *           printf("RC5 %s t=%ld a=%ld c=%ld\n",
 *               binary_string(Rc5_GetCode(&rc5)),
 *               Rc5_GetToggle(&rc5),
 *               Rc5_GetAddress(&rc5),
 *               Rc5_GetCommand(&rc5));
 *       }
 *
 * See libtest.c for more complete details.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include "drivers/lirc.h"
#include "rc5.h"

#undef DEBUG

/*----------------------------------------------------------
 *  Timing Constants
 *---------------------------------------------------------*/

/*
 * Timing is derived from a 432kHz clock.
 * A bit time is 3 * 2^8 clock periods or 1.778ms.
 */

#define MEAN_LONG_PULSE  1778 
#define MEAN_SHORT_PULSE (MEAN_LONG_PULSE/2)

/*
 * We define an acceptance tolerance of 1/4 of a long pulse
 */
#define PULSE_MARGIN     (MEAN_LONG_PULSE/4)

/*
 * Now we define min and max values for short and long pulses
 */
#define MIN_SHORT_PULSE  (MEAN_SHORT_PULSE - PULSE_MARGIN)
#define MAX_SHORT_PULSE  (MEAN_SHORT_PULSE + PULSE_MARGIN)
#define MIN_LONG_PULSE   (MEAN_LONG_PULSE  - PULSE_MARGIN)
#define MAX_LONG_PULSE   (MEAN_LONG_PULSE  + PULSE_MARGIN)

/*----------------------------------------------------------
 *  Bit Operation Definitions
 *  Used for extracting fields from the RC5 14-bit word.
 *---------------------------------------------------------*/

#define RC5_TOTAL_BITS 14
/*
 * NBITS(N) returns a value with N low bits set
 */
#define NBITS(N) ((1<<N)-1)
/*
 * MASK(BITS, SHIFT) returns a bitmask with BITS bits 
 * set, shifted left by SHIFT bits.
 */
#define MASK(BITS, SHIFT) (NBITS(BITS)<<SHIFT)

/*
 * The command is the low 6 bits
 * --------xxxxxx
 */
#define COMMAND_SHIFT 0
#define COMMAND_BITS  6
#define COMMAND_MASK MASK(COMMAND_BITS, COMMAND_SHIFT)

/*
 * The address is 5 bits starting at bit 6
 * ---xxxxx------
 */
#define ADDRESS_SHIFT 6
#define ADDRESS_BITS  5
#define ADDRESS_MASK MASK(ADDRESS_BITS, ADDRESS_SHIFT)

/*
 * The toggle is 1 bit at bit 11
 * --x-----------
 */
#define TOGGLE_SHIFT 11
#define TOGGLE_BITS   1
#define TOGGLE_MASK MASK(TOGGLE_BITS, TOGGLE_SHIFT)

/*
 * The S2 is 1 bit at bit 12
 * -x------------
 * Stop bit 2 is used in extended RC5 to extend the 
 * command set.  Stop bit 1 is always 1.
 */
#define S2_SHIFT     12
#define S2_BITS       1
#define S2_MASK MASK(S2_BITS, S2_SHIFT)

/* to shift S2 down to top of COMMAND (=6) */
#define XCOMMAND_SHIFT (S2_SHIFT - COMMAND_BITS)

/*----------------------------------------------------------
 *  State Machine Definitions
 *---------------------------------------------------------*/

/*
 * Every pulse is defined by two attributes:
 * Bit 0 : Duration : Short=0, Long =1
 * Bit 1 : State :    Space=0, Pulse=1
 */
#define SHORT_ATTR 0
#define LONG_ATTR  1
#define SPACE_ATTR 0
#define PULSE_ATTR 2

typedef enum {
  SHORT_SPACE = (SHORT_ATTR | SPACE_ATTR),
  LONG_SPACE  = (LONG_ATTR  | SPACE_ATTR),
  SHORT_PULSE = (SHORT_ATTR | PULSE_ATTR),
  LONG_PULSE  = (LONG_ATTR  | PULSE_ATTR)
} RC5_EVENT;

/*
 * Give each of the events names.
 */
static const char *EventName[] = {
  "SHORT_SPACE",
  "LONG_SPACE",
  "SHORT_PULSE",
  "LONG_PULSE"
};

/*  
 * Give each of the states names
 */
static const char *StateName[] =
{
  "ERROR",
  "MID0",
  "MID1",
  "START0",
  "START1"
};

/*
 * Define the possible emissions.
 */

typedef enum {
  RC5_0 = 0,
  RC5_1 = 1,
  RC5_X = 2   /* no emission */
} RC5_EMIT;

/*
 * Name the possible emissions.
 */

char *EmitName[] =
{
  "0",
  "1",
  "X"
};

/*
 * Define a struct to describe a transition
 */

typedef struct {
  RC5_STATE         FinalState;
  RC5_EMIT          Emit;
} RC5_TRANSITION;

/*
 * Finally we define the transition table itself.
 */

RC5_TRANSITION TransitionTable[5][4] =
{
  {                           /* from ERROR state */
    {RC5_ERROR,   RC5_X},
    {RC5_ERROR,   RC5_X},
    {RC5_ERROR,   RC5_X},
    {RC5_ERROR,   RC5_X} 
  },
  {                           /* MID0 state   */
    {RC5_START0,  RC5_X},     /* SHORT_SPACE  */
    {RC5_MID1,    RC5_1},     /* LONG_SPACE   */
    {RC5_ERROR,   RC5_X},     /* SHORT_PULSE  */
    {RC5_ERROR,   RC5_X}      /* LONG_PULSE   */
  },
  {                           /* MID1 state   */
    {RC5_ERROR,   RC5_X},     /* SHORT_SPACE  */
    {RC5_ERROR,   RC5_X},     /* LONG_SPACE   */
    {RC5_START1,  RC5_X},     /* SHORT_PULSE  */
    {RC5_MID0,    RC5_0}      /* LONG_PULSE   */
  },
  {                           /* START0 state */
    {RC5_ERROR,   RC5_X},     /* SHORT_SPACE  */
    {RC5_ERROR,   RC5_X},     /* LONG_SPACE   */
    {RC5_MID0,    RC5_0},     /* SHORT_PULSE  */
    {RC5_ERROR,   RC5_X}      /* LONG_PULSE   */
  },
  {                           /* START1 state */
    {RC5_MID1,    RC5_1},     /* SHORT_SPACE  */
    {RC5_ERROR,   RC5_X},     /* LONG_SPACE   */
    {RC5_ERROR,   RC5_X},     /* SHORT_PULSE  */
    {RC5_ERROR,   RC5_X}      /* LONG_PULSE   */
  }
};


/*----------------------------------------------------------
 *  Decoding Functions
 *---------------------------------------------------------*/

/*
 * Initialize the RC5 state.
 */
void Rc5_Init (RC5_CONTEXT *Rc5)
{
  Rc5->Value = 1;
  Rc5->State = RC5_MID1;
  Rc5->Bits  = 1;
}

/*
 * Rc5_AddBit
 * 
 * Internal.
 * Called when the state machine wants to shift a bit
 * into the RC5 14-bit word.  Sets the error state
 * if the word already contains 14-bits.
 * 
 */
void Rc5_AddBit (RC5_CONTEXT *Rc5, RC5_EMIT Emit)
{
    if (Emit!=RC5_X) {
        if (Rc5->Bits > 14) {
            Rc5->State = RC5_ERROR;
        } else {
            Rc5->Value<<=1;
            if (Emit==RC5_1) {
                Rc5->Value |= 1;
            }
            Rc5->Bits++;
        }
    }
}

/*
 * Rc5_StateChange
 * 
 * Called when a new pulse or space is detected.
 * Advances the state machine.
 */
RC5_STATE Rc5_StateChange (RC5_CONTEXT *Rc5, RC5_EVENT Event)
{
  RC5_TRANSITION *transition = &(TransitionTable[Rc5->State][Event]);

#ifdef DEBUG
  printf ("%s : %s -> %s [%s] = ",
	  EventName[Event],
	  StateName[Rc5->State],
	  StateName[transition->FinalState],
	  EmitName[transition->Emit]);
#endif

  Rc5->State = transition->FinalState;
  Rc5_AddBit(Rc5, transition->Emit);

#ifdef DEBUG
  printf ("= %lx/%d\n", Rc5->Value, Rc5->Bits);
#endif

  return Rc5->State;
}

/*
 * Rc5_Done
 * 
 * Returns a non-zero value if a complete RC5 code word has
 * been decoded.
 */
unsigned char Rc5_Done (RC5_CONTEXT *Rc5)
{
  return (Rc5->Bits==RC5_TOTAL_BITS) &&
    (Rc5->State == RC5_START1 ||  /* consume final pulse */
     Rc5->State == RC5_MID0);
}

/*
 * Rc5_GetCode
 * 
 * Returns the complete 14-bit code word.
 */
unsigned long Rc5_GetCode (RC5_CONTEXT *Rc5)
{
  return Rc5->Value;
}

/*
 * Rc5_GetToggle
 * 
 * Returns the value field from a decoded code word.
 */
unsigned long Rc5_GetToggle (RC5_CONTEXT *Rc5)
{
  return (Rc5->Value & TOGGLE_MASK) >> TOGGLE_SHIFT;
}

/*
 * Rc5_GetAddress
 * 
 * Returns the address field from a decoded code word.
 */
unsigned long Rc5_GetAddress (RC5_CONTEXT *Rc5)
{
  return (Rc5->Value & ADDRESS_MASK) >> ADDRESS_SHIFT;
}

/*
 * Rc5_GetCommand
 * 
 * Returns the command field from a decoded code word.
 */
unsigned long Rc5_GetCommand (RC5_CONTEXT *Rc5)
{
  return (Rc5->Value & COMMAND_MASK) >> COMMAND_SHIFT;
}

/*
 * Rc5_GetExtCommand
 * 
 * Returns the extended command field from a decoded code word.
 */
unsigned long Rc5_GetExtCommand (RC5_CONTEXT *Rc5)
{
  return Rc5_GetCommand(Rc5) | ((Rc5->Value & S2_MASK) >> XCOMMAND_SHIFT);
}


/*----------------------------------------------------------
 * LIRC Interface
 *---------------------------------------------------------*/


/*
 * Rc5_Notify
 * 
 * LIRC specific:
 * Called with the lirc signal which contains a 
 * pulse/space information.
 *
 * Returns 1 if a code word is available, 0 otherwise.
 */
char Rc5_Notify (RC5_CONTEXT *Rc5, lirc_t Signal)
{
    char done = 0;
    char err = 0;
    RC5_EVENT event;
    long duration;
    
    /* 
     * reinitialize if the last code word was complete.
     */
    if (Rc5_Done(Rc5)) {
        Rc5_Init(Rc5);
    }
    
    /*
     * determine the event code from the lirc event.
     */
    duration = Signal & PULSE_MASK;
    event = (Signal & PULSE_BIT) ? PULSE_ATTR : SPACE_ATTR;
    
    if (duration >= MIN_SHORT_PULSE &&
        duration <= MAX_SHORT_PULSE) {
        event |= SHORT_ATTR;
    } else if (duration >= MIN_LONG_PULSE &&
        duration <= MAX_LONG_PULSE) {
        event |= LONG_ATTR;
    } else {
#ifdef DEBUG
        printf("Duration %ld out of range\n",duration);
#endif
        err++;
    }
    
    /* 
     * If the signal was within spec, 
     * advance the state machine.
     */
    if (!err) {
        err = (Rc5_StateChange(Rc5, event)==RC5_ERROR);
    }
    
    /* 
     * If no error occured, check to see if the code word is complete. 
     * If an error occured, we reset the state machine.
     */
    if (!err) {
        if (Rc5_Done(Rc5)) {
            done = 1;
        }
    } else {
        Rc5_Init(Rc5);
    }
    
    return done;
}
